"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGroup = createGroup;
exports.updateGroup = updateGroup;
exports.getGroups = getGroups;
exports.getGroupsWithOutPagination = getGroupsWithOutPagination;
exports.deleteGroup = deleteGroup;
exports.getGroupById = getGroupById;
const common_service_1 = require("../../service/common.service");
const logger_service_1 = require("../../service/logger.service");
const api_response_1 = require("../../api.response");
const common_service_2 = require("../../service/common.service");
const group_entity_1 = __importDefault(require("../entities/group.entity"));
const mongoose_1 = __importDefault(require("mongoose"));
function createGroup(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { groupName, ageGroup, subscriptionPlan, transcribeUsageRange, users, status, } = req.body;
        if (!groupName ||
            !ageGroup ||
            !subscriptionPlan ||
            !transcribeUsageRange ||
            !users) {
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.allFieldsRequired);
        }
        try {
            const existingGroup = yield group_entity_1.default.findOne({
                groupName,
                isDeleted: false,
            });
            if (existingGroup) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.group.groupExits);
            }
            else {
                const savedGroup = yield group_entity_1.default.create({
                    groupName,
                    ageGroup,
                    subscriptionPlan,
                    transcribeUsageRange,
                    users,
                    status,
                });
                yield savedGroup.populate({
                    path: "users",
                    select: "-password",
                });
                yield savedGroup.populate("users", "-password");
                return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Create, api_response_1.messages.group.success, savedGroup);
            }
        }
        catch (error) {
            console.log("Getting error while create group : ", error);
            logger_service_1.logger.error(`Getting error while create group : ${error}`);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
// Update an existing Group
function updateGroup(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { groupName, ageGroup, subscriptionPlan, transcribeUsageRange, users, status, _id, } = req.body;
        try {
            const updatedGroup = yield group_entity_1.default.findOneAndUpdate({ _id: new mongoose_1.default.Types.ObjectId(_id), isDeleted: false }, {
                groupName,
                ageGroup,
                subscriptionPlan,
                transcribeUsageRange,
                users,
                status,
            }, { new: true });
            if (!updatedGroup) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.NotFound, api_response_1.messages.group.groupNotFound);
            }
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.group.groupUpdated, updatedGroup);
        }
        catch (error) {
            logger_service_1.logger.error("Getting error while update group : ", error);
            console.log("Getting error while update group : ", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
// ===================== GET GROUP =========================//
function getGroups(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const query = req.query;
            const { search, page, take, role } = query;
            // Build search query
            const searchQuery = {};
            if (search) {
                searchQuery.groupName = { $regex: search, $options: "i" };
            }
            let filters = { isDeleted: false };
            if (role)
                filters.role = role;
            const options = {
                searchQuery,
                filter: filters,
                page: parseInt(page) || 1,
                take: parseInt(take) || 10,
            };
            const { data: groups, totalData, page: currentPage, take: perPage, totalPages, prevPage, nextPage, } = yield (0, common_service_2.fetchUsersWithFilterAndPagination)(group_entity_1.default, options);
            const populatedGroup = yield group_entity_1.default.populate(groups, {
                path: "users",
                select: "username",
            });
            if (!groups.length) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.noDataAvailable);
            }
            const responseData = {
                data: populatedGroup || [],
                pagination: {
                    nextPage,
                    page: currentPage,
                    prevPage,
                    take: perPage,
                    totalData,
                    totalPages,
                },
            };
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.success, responseData);
        }
        catch (error) {
            logger_service_1.logger.error(`Getting error while get groups : ${error}`);
            console.error("Getting error while get groups : ", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
// ===================== GET GROUP WITHOUT PAGINATION ===============//
function getGroupsWithOutPagination(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const data = yield group_entity_1.default.aggregate([
                { $match: { isDeleted: false } },
                { $sort: { createdAt: -1 } },
                {
                    $lookup: {
                        from: "users",
                        localField: "users",
                        foreignField: "_id",
                        as: "userDetails",
                    },
                },
                {
                    $addFields: {
                        userDetails: {
                            $map: {
                                input: "$userDetails",
                                as: "user",
                                in: { _id: "$$user._id" },
                            },
                        },
                    },
                },
                {
                    $project: {
                        _id: 1,
                        groupName: 1,
                        ageGroup: 1,
                        transcribeUsageRange: 1,
                        subscriptionPlan: 1,
                        userDetails: 1,
                        status: 1,
                        isDeleted: 1,
                        createdAt: 1,
                        updatedAt: 1,
                        __v: 1,
                    },
                },
            ]);
            if (!data.length) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.noDataAvailable);
            }
            const responseData = {
                data: data || [],
            };
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.success, responseData);
        }
        catch (error) {
            logger_service_1.logger.error(`Getting error while get groups with out pagination : ${error}`);
            console.log("Getting error while get groups with out pagination : ", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
// ===================== DELETE A GROUP ===============//
function deleteGroup(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { id } = req.params;
        try {
            yield group_entity_1.default.findOneAndUpdate({ _id: new mongoose_1.default.Types.ObjectId(id), isDeleted: false }, { isDeleted: true });
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.group.groupDeleted);
        }
        catch (error) {
            logger_service_1.logger.error(`Getting error while delete group : ${error}`);
            console.log("Getting error while delete group : ", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
// ===================== GET A GROUP BY ID ===============//
function getGroupById(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const group = yield group_entity_1.default.aggregate([
                {
                    $match: {
                        _id: new mongoose_1.default.Types.ObjectId(req.params.id),
                        isDeleted: false,
                    },
                },
                {
                    $lookup: {
                        from: "users",
                        localField: "users",
                        foreignField: "_id",
                        as: "userDetails",
                    },
                },
                {
                    $addFields: {
                        userDetails: {
                            $map: {
                                input: "$userDetails",
                                as: "user",
                                in: { _id: "$$user._id" },
                            },
                        },
                    },
                },
                {
                    $project: {
                        _id: 1,
                        groupName: 1,
                        ageGroup: 1,
                        transcribeUsageRange: 1,
                        subscriptionPlan: 1,
                        userDetails: 1,
                        status: 1,
                        isDeleted: 1,
                        createdAt: 1,
                        updatedAt: 1,
                        __v: 1,
                    },
                },
            ]);
            if (group.length === 0) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.NotFound, api_response_1.messages.group.groupNotFound);
            }
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.success, group[0]);
        }
        catch (error) {
            logger_service_1.logger.error(`Getting error while get group by id : ${error}`);
            console.log("Getting error while get group by id : ", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
