"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.jsonWebToken = jsonWebToken;
exports.signUp = signUp;
exports.signIn = signIn;
exports.verifyOTP = verifyOTP;
exports.resendOTP = resendOTP;
exports.resetPassword = resetPassword;
exports.changePassword = changePassword;
exports.logOut = logOut;
exports.googleAuth = googleAuth;
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const bcrypt_1 = __importDefault(require("bcrypt"));
const dotenv_1 = __importDefault(require("dotenv"));
const common_service_1 = require("../../service/common.service");
const moment_1 = __importDefault(require("moment"));
const api_response_1 = require("../../api.response");
const logger_service_1 = require("../../service/logger.service");
const app_user_entity_1 = __importDefault(require("../../user/entities/app-user.entity"));
const users_entity_1 = __importDefault(require("../../user/entities/users.entity"));
const admin_user_entity_1 = __importDefault(require("../../user/entities/admin-user.entity"));
const auth_token_entity_1 = __importDefault(require("../entities/auth-token.entity"));
const utills_1 = require("../../service/utills");
const mongoose_1 = __importDefault(require("mongoose"));
const workspace_entity_1 = __importDefault(require("../../workspace/entities/workspace.entity"));
dotenv_1.default.config();
const secret_key = process.env.JWT_SECRET || "";
function jsonWebToken(payload_1) {
    return __awaiter(this, arguments, void 0, function* (payload, expiresIn = "365d") {
        try {
            const token = jsonwebtoken_1.default.sign(payload, secret_key, {
                algorithm: "HS256", // Use a proper algorithm here
                expiresIn,
            });
            return token;
        }
        catch (error) {
            console.error("Error while creating JWT token:", error);
            logger_service_1.logger.error(`Error while creating JWT token: ${error}`);
            return undefined;
        }
    });
}
//************* APP USER SIGN-UP ********************//
function signUp(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { firstName, lastName, phone, email, password } = req.body;
        // Validate required fields
        if (!email) {
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.allFieldsRequired);
        }
        try {
            // // Check if user already exists
            const existingUser = yield users_entity_1.default.findOne({
                email,
                isDeleted: false,
            });
            // If an existing user is found
            if (existingUser) {
                // Handle unverified users (AppUser)
                const existingAppUser = yield app_user_entity_1.default.findOne({
                    email,
                    isDeleted: false,
                });
                if (existingAppUser && !existingAppUser.isVerify) {
                    // const otp = Math.floor(1000 + Math.random() * 9000).toString();
                    const otp = 1234;
                    existingUser.otp = otp;
                    yield existingUser.save();
                    // const emailSent = await sendEmail({
                    //   userEmail: existingUser.email,
                    //   subject: "Your OTP Code",
                    //   text: `Your OTP code is ${otp}. Please use this code to verify your email.`,
                    //   html: `<p>Your OTP code is <strong>${otp}</strong>. Please use this code to verify your email.</p>`,
                    // });
                    // if (!emailSent) {
                    //   return sendMessageResponse(
                    //     req,
                    //     res,
                    //     StatusCode.InternalServerError,
                    //     "Error sending verification email"
                    //   );
                    // }
                    return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.signIn.otpSent, Object.assign({ otp }, existingAppUser._doc));
                }
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.signUp.emailTaken);
            }
            const bcryptStr = yield bcrypt_1.default.hash(password, 12);
            // Create the new user
            const newUser = new users_entity_1.default({
                email,
                password: bcryptStr,
                role: utills_1.ROLES.USER,
                userType: utills_1.USER_TYPE.USER,
                status: true,
                isDeleted: false,
            });
            const savedUser = yield newUser.save();
            yield savedUser.populate("role", "roleName _id");
            // Save User
            let fullName = firstName + " " + lastName;
            const appUser = new app_user_entity_1.default({
                firstName,
                lastName,
                fullName,
                email,
                userId: savedUser === null || savedUser === void 0 ? void 0 : savedUser._id,
                phone,
                userType: utills_1.USER_TYPE.USER,
            });
            yield appUser.save();
            const untitledWorkspace = new workspace_entity_1.default({
                name: "Untitled Workspace",
                users: [savedUser === null || savedUser === void 0 ? void 0 : savedUser._id],
                owner: savedUser === null || savedUser === void 0 ? void 0 : savedUser._id,
            });
            yield untitledWorkspace.save();
            // Send OTP for email verification
            // const otp = Math.floor(1000 + Math.random() * 9000).toString();
            const otp = 1234;
            newUser.otp = otp;
            yield newUser.save();
            // const emailSent = await sendEmail({
            //   userEmail: newUser.email,
            //   subject: "Your OTP Code",
            //   text: `Your OTP code is ${otp}. Please use this code to verify your email.`,
            //   html: `<p>Your OTP code is <strong>${otp}</strong>. Please use this code to verify your email.</p>`,
            // });
            // if (!emailSent) {
            //   return sendMessageResponse(
            //     req,
            //     res,
            //     StatusCode.InternalServerError,
            //     "Error sending verification email"
            //   );
            // }
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.signIn.otpSent, Object.assign({ otp }, appUser._doc));
        }
        catch (error) {
            logger_service_1.logger.error(`Getting error while add-app-user: ${error}`);
            console.error("Getting error while add-app-user: ", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//************* SIGN-IN ********************//
function signIn(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { email, password } = req.body;
        if (!email || !password) {
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.allFieldsRequired);
        }
        try {
            const user = yield users_entity_1.default.findOne({ email, isDeleted: false });
            if (!user || !user.password) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.signIn.invalidCredentials);
            }
            const isPasswordValid = yield bcrypt_1.default.compare(password, user === null || user === void 0 ? void 0 : user.password);
            if (!isPasswordValid) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.signIn.invalidCredentials);
            }
            const tokenpayload = {
                _id: user._id,
                email: user.email,
                userType: user.userType,
                status: user.status,
                roleId: user.role,
            };
            const token = yield jsonWebToken(tokenpayload);
            let users;
            let permission;
            yield auth_token_entity_1.default.create({
                userId: user === null || user === void 0 ? void 0 : user._id,
                token: token,
            });
            if (user === null || user === void 0 ? void 0 : user.status) {
                if ((user === null || user === void 0 ? void 0 : user.userType) == utills_1.USER_TYPE.SUPER_ADMIN ||
                    (user === null || user === void 0 ? void 0 : user.userType) == utills_1.USER_TYPE.SUB_ADMIN) {
                    const users = yield admin_user_entity_1.default
                        .aggregate([
                        {
                            $match: {
                                userId: new mongoose_1.default.Types.ObjectId(user === null || user === void 0 ? void 0 : user._id),
                            },
                        },
                        {
                            $lookup: {
                                from: "roles",
                                localField: "role",
                                foreignField: "_id",
                                as: "roleDetails",
                            },
                        },
                        {
                            $unwind: {
                                path: "$roleDetails",
                            },
                        },
                        {
                            $addFields: {
                                roleName: { $ifNull: ["$roleDetails.roleName", null] }, // Add roleName to top level
                                permission: { $ifNull: ["$roleDetails.permission", null] }, // Add permission to top level
                            },
                        },
                        {
                            $project: {
                                _id: 1,
                                userId: 1,
                                profileImage: 1,
                                email: 1,
                                firstName: 1,
                                lastName: 1,
                                fullName: 1,
                                phone: 1,
                                status: 1,
                                role: 1,
                                userType: 1,
                                __v: 1,
                                roleName: 1,
                                permission: 1,
                            },
                        },
                    ])
                        .exec();
                    if (users.length) {
                        return (0, common_service_1.sendTokenResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.signIn.success, users[0], token);
                    }
                    else {
                        return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Unauthorized, api_response_1.messages.unauthorizedUser);
                    }
                }
                else {
                    users = yield app_user_entity_1.default.findOne({ userId: user === null || user === void 0 ? void 0 : user._id });
                    if (users === null || users === void 0 ? void 0 : users.isVerify) {
                        users.loginCount += 1;
                        yield users.save();
                        return (0, common_service_1.sendTokenResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.signIn.success, Object.assign(Object.assign({}, users._doc), { permission }), token);
                    }
                    else {
                        // const otp = Math.floor(1000 + Math.random() * 9000).toString();
                        const otp = 1234;
                        user.otp = otp;
                        yield user.save();
                        // const emailSent = await sendEmail({
                        //   userEmail: user.email,
                        //   subject: "Your OTP Code",
                        //   text: `Your OTP code is ${otp}. Please use this code to verify your email.`,
                        //   html: `<p>Your OTP code is <strong>${otp}</strong>. Please use this code to verify your email.</p>`,
                        // });
                        // if (!emailSent) {
                        //   return sendMessageResponse(
                        //     req,
                        //     res,
                        //     StatusCode.InternalServerError,
                        //     messages.signIn.otpSentFail
                        //   );
                        // }
                        return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.signIn.otpSent, Object.assign({ otp }, users._doc));
                    }
                }
            }
            else {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Unauthorized, api_response_1.messages.unauthorizedUser);
            }
        }
        catch (error) {
            console.error("Error during sign-in:", error);
            logger_service_1.logger.error(`Getting error while sign-In : ${error}`);
            return yield (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.error);
        }
    });
}
//*************verify-OTP********************//
function verifyOTP(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { email, otp } = req.body;
        if (!email || !otp) {
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.allFieldsRequired);
        }
        try {
            const user = yield users_entity_1.default.findOne({ email, isDeleted: false });
            if (!user || user.otp !== otp) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Unauthorized, api_response_1.messages.verifyOTP.invalidOTP);
            }
            // Clear OTP from user document
            user.otp = null;
            yield user.save();
            const tokenpayload = {
                _id: user._id,
                email: user.email,
                userType: user.userType,
                status: user.status,
                roleId: user.role,
            };
            const token = yield jsonWebToken(tokenpayload, "2h");
            yield auth_token_entity_1.default.create({
                userId: user === null || user === void 0 ? void 0 : user._doc._id,
                token, // Ensure this is a string
            });
            if (user._doc.userType == utills_1.USER_TYPE.SUB_ADMIN) {
                return (0, common_service_1.sendTokenResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.verifyOTP.success, user._doc, token);
            }
            else {
                const appUser = yield app_user_entity_1.default.findOne({
                    userId: user._doc._id,
                });
                appUser.isVerify = true;
                yield appUser.save();
                return (0, common_service_1.sendTokenResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.verifyOTP.success, Object.assign({}, appUser._doc), token);
            }
        }
        catch (error) {
            console.error("Error during OTP verification:", error);
            logger_service_1.logger.error(`Getting error while OTP verification : ${error}`);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//*************resend-OTP********************//
function resendOTP(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { email } = req.body;
        const currentTime = (0, moment_1.default)();
        if (!email) {
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.allFieldsRequired);
        }
        try {
            const user = yield users_entity_1.default.findOne({ email, isDeleted: false });
            if (!user) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Unauthorized, api_response_1.messages.noUsersFound);
            }
            if (user) {
                if (user.otpSentTime) {
                    const dateTimeDiff = currentTime.diff((0, moment_1.default)(user.otpSentTime), "minutes");
                    if (dateTimeDiff < 2) {
                        return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.TooManyConnection, api_response_1.messages.resendOTP.tooSoon);
                    }
                }
            }
            // Generate new OTP
            // const otp = Math.floor(1000 + Math.random() * 9000).toString();
            const otp = 1234;
            user.otp = otp;
            yield user.save();
            // const emailSent = await sendEmail({
            //   userEmail: user.email,
            //   subject: "Your OTP Code",
            //   text: `Your OTP code is ${otp}. Please use this code to verify your email.`,
            //   html: `<p>Your OTP code is <strong>${otp}</strong>. Please use this code to verify your email.</p>`,
            // });
            // if (!emailSent) {
            //   return sendMessageResponse(
            //     req,
            //     res,
            //     StatusCode.InternalServerError,
            //     "Error sending verification email"
            //   );
            // }
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.signIn.otpSent, {
                otp,
                email,
            });
        }
        catch (error) {
            console.error("Error during OTP resend:", error);
            logger_service_1.logger.error(`Getting error while OTP resend : ${error}`);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//*************reset-password********************//
function resetPassword(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { password, resetToken } = req.body;
        try {
            const decoded = jsonwebtoken_1.default.verify(resetToken, secret_key);
            if (decoded) {
                const user = yield users_entity_1.default.findOne({
                    _id: new mongoose_1.default.Types.ObjectId(decoded === null || decoded === void 0 ? void 0 : decoded._id),
                    isDeleted: false,
                });
                if (!user) {
                    return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.signIn.invalidCredentials);
                }
                const hashedPassword = yield bcrypt_1.default.hash(password, 10);
                // Update user's password with the new hashed password
                user.password = hashedPassword;
                yield user.save();
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.resetPassword.success);
            }
            else {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Unauthorized, api_response_1.messages.resetPassword.invalidToken);
            }
        }
        catch (error) {
            logger_service_1.logger.error(`Getting error while reset password : ${error}`);
            console.log("Getting error while reset password : ", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//*************change-password********************//
function changePassword(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { userId, currentPassword, newPassword } = req.body;
        if (!userId || !currentPassword || !newPassword) {
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.allFieldsRequired);
        }
        try {
            const user = yield users_entity_1.default.findById(userId);
            if (!user) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.NotFound, api_response_1.messages.noUsersFound);
            }
            const isPasswordMatch = yield bcrypt_1.default.compare(currentPassword, user.password);
            if (!isPasswordMatch) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Unauthorized, api_response_1.messages.incorrectPassword);
            }
            const hashedNewPassword = yield bcrypt_1.default.hash(newPassword, 10);
            user.password = hashedNewPassword;
            yield user.save();
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.changePassword.success);
        }
        catch (error) {
            logger_service_1.logger.error(`Error changing password: ${error}`);
            console.error("Error changing password:", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//====================  LOGGED OUT ========================//
function logOut(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a, _b;
        try {
            const token = (_b = (_a = req === null || req === void 0 ? void 0 : req.headers) === null || _a === void 0 ? void 0 : _a.authorization) === null || _b === void 0 ? void 0 : _b.split(" ")[1];
            const userToken = yield auth_token_entity_1.default.findOne({
                token: token,
            });
            yield auth_token_entity_1.default.deleteMany({
                userId: userToken === null || userToken === void 0 ? void 0 : userToken.userId,
            });
            return yield (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.logged_out);
        }
        catch (error) {
            console.log("Getting error while logged out :", error);
            logger_service_1.logger.error(`Getting error while logged out : ${error}`);
            return yield (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.error);
        }
    });
}
//====================  GOOGLE AUTH ========================//
function googleAuth(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const { email } = req.body;
            // Check if the email is provided
            if (!email) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.allFieldsRequired);
            }
            // Check if the user already exists
            const existingUser = yield users_entity_1.default.findOne({ email });
            let appUser;
            let userId;
            if (existingUser) {
                // Existing user found, update appUser details
                appUser = yield app_user_entity_1.default.findOne({ userId: existingUser._id });
                appUser.isVerify = true;
                appUser.loginCount += 1;
                userId = existingUser._id;
            }
            else {
                // Create new user and corresponding appUser
                const savedUser = yield users_entity_1.default.create({
                    email,
                    status: true,
                });
                userId = savedUser._id;
                // Save the appUser
                appUser = yield app_user_entity_1.default.create({
                    userId: savedUser._id,
                    email: savedUser.email,
                    userType: savedUser.userType,
                    status: savedUser.status,
                    isVerify: true,
                    loginCount: 1,
                });
            }
            // Generate JWT token
            const tokenpayload = {
                _id: userId,
                email: appUser.email,
                userType: appUser.userType,
                status: appUser.status,
                roleId: appUser.role,
            };
            const token = yield jsonWebToken(tokenpayload);
            // Store the token in AuthTokenSchema
            yield auth_token_entity_1.default.create({ userId: userId, token });
            // Send the response
            return (0, common_service_1.sendTokenResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.signIn.success, appUser, token);
        }
        catch (error) {
            logger_service_1.logger.error(`Error during Google Authentication : ${error}`);
            console.error("Error during Google Authentication:", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
